/**
 * Universal Test Runner for Learnlytica Assessment Pro
 *
 * This script reads learnlytica.json configuration and executes tests
 * according to the defined components and scripts.
 */

const fs = require('fs');
const path = require('path');
const { spawn, execFileSync } = require('child_process');

// Try to load glob - it's optional for path expansion
let glob = null;
try {
    glob = require('glob');
} catch (e) {
    console.warn('[UniversalRunner] glob module not found. Path glob expansion will be disabled. Run: npm install glob');
}

// Configuration file name
const CONFIG_FILE = 'learnlytica.json';

// Default configuration values
const DEFAULTS = {
    version: '1.0',
    testReportFile: 'test_report.log',
    specificationFile: 'specification.md',
    pipeline: 'sequential'
};

const COMPONENT_DEFAULTS = {
    prerequisites: [],
    scripts: {
        preInstall: '',
        install: '',
        postInstall: '',
        build: '',
        test: '',
        run: ''
    },
    env: {}
};

/**
 * Parse JDK version from directory name for proper sorting.
 */
function parseJdkVersion(dirName) {
    const basename = path.basename(dirName);
    const versionMatch = basename.match(/(?:jdk|java|openjdk|temurin|corretto|zulu|adoptium)[-_]?(\d+(?:[._]\d+)*)/i);
    if (versionMatch && versionMatch[1]) {
        return versionMatch[1].split(/[._]/).map(Number);
    }
    return null;
}

/**
 * Compare two JDK version arrays.
 */
function compareJdkVersions(a, b) {
    if (a === null && b === null) return 0;
    if (a === null) return -1;
    if (b === null) return 1;
    const maxLen = Math.max(a.length, b.length);
    for (let i = 0; i < maxLen; i++) {
        const va = a[i] || 0;
        const vb = b[i] || 0;
        if (va !== vb) return va - vb;
    }
    return 0;
}

/**
 * Check if a path contains a valid Java installation
 */
function isValidJavaHome(javaHomePath) {
    if (!javaHomePath || !fs.existsSync(javaHomePath)) return false;
    const javaExe = process.platform === 'win32'
        ? path.join(javaHomePath, 'bin', 'java.exe')
        : path.join(javaHomePath, 'bin', 'java');
    return fs.existsSync(javaExe);
}

/**
 * Scan a directory for JDK installations and return the highest version found.
 */
function findHighestJavaInDir(baseDir, filterFn = null) {
    if (!fs.existsSync(baseDir)) return undefined;
    try {
        let dirs = fs.readdirSync(baseDir).filter(d => !d.startsWith('.'));
        if (filterFn) dirs = dirs.filter(filterFn);
        dirs.sort((a, b) => compareJdkVersions(parseJdkVersion(b), parseJdkVersion(a)));
        for (const dir of dirs) {
            const candidate = path.join(baseDir, dir);
            if (isValidJavaHome(candidate)) return candidate;
        }
    } catch (e) {
        if (e.code !== 'ENOENT' && e.code !== 'ENOTDIR') {
            console.warn(`[findHighestJavaInDir] Failed to scan ${baseDir}: ${e.message}`);
        }
    }
    return undefined;
}

/**
 * Try to derive JAVA_HOME from the java command in PATH.
 */
function deriveJavaHomeFromPath() {
    try {
        let javaPath;
        if (process.platform === 'win32') {
            const output = execFileSync('where', ['java'], { encoding: 'utf8', stdio: ['pipe', 'pipe', 'pipe'], timeout: 5000 });
            javaPath = (output || '').split('\n')[0].trim();
        } else {
            const output = execFileSync('which', ['java'], { encoding: 'utf8', stdio: ['pipe', 'pipe', 'pipe'], timeout: 5000 });
            javaPath = (output || '').trim();
        }
        if (javaPath && fs.existsSync(javaPath)) {
            const realPath = fs.realpathSync(javaPath);
            const binDir = path.dirname(realPath);
            const javaHome = path.dirname(binDir);
            if (isValidJavaHome(javaHome)) return javaHome;
        }
    } catch (e) {
        if (e.code !== 'ENOENT' && !e.message.includes('not found')) {
            console.warn(`[deriveJavaHomeFromPath] Could not derive JAVA_HOME from PATH: ${e.message}`);
        }
    }
    return undefined;
}

/**
 * Check Java version managers for valid installations.
 */
function checkVersionManagers(homeDir) {
    if (!homeDir) return undefined;
    const directPaths = [
        path.join(homeDir, '.sdkman', 'candidates', 'java', 'current'),
        path.join(homeDir, '.jabba', 'jdk', 'default'),
    ];
    for (const p of directPaths) {
        if (isValidJavaHome(p)) return p;
    }
    const asdfJavaDir = path.join(homeDir, '.asdf', 'installs', 'java');
    const asdfResult = findHighestJavaInDir(asdfJavaDir);
    if (asdfResult) return asdfResult;
    return undefined;
}

/**
 * Find JAVA_HOME by checking multiple sources in order of priority.
 */
function findJavaHome() {
    if (process.env.JAVA_HOME && isValidJavaHome(process.env.JAVA_HOME)) {
        return process.env.JAVA_HOME;
    }
    const pathDerived = deriveJavaHomeFromPath();
    if (pathDerived) return pathDerived;
    const homeDir = process.env.HOME || process.env.USERPROFILE || '';
    const vmResult = checkVersionManagers(homeDir);
    if (vmResult) return vmResult;
    if (process.platform === 'win32') {
        const candidates = [
            process.env.ProgramFiles && path.join(process.env.ProgramFiles, 'Java'),
            process.env['ProgramFiles(x86)'] && path.join(process.env['ProgramFiles(x86)'], 'Java'),
            'C:\\Program Files\\Java',
            'C:\\Program Files (x86)\\Java',
        ].filter(Boolean);
        for (const base of candidates) {
            const result = findHighestJavaInDir(base, d => d.toLowerCase().startsWith('jdk') || d.toLowerCase().includes('java'));
            if (result) return result;
        }
    } else {
        const staticCandidates = [
            '/opt/homebrew/opt/openjdk',
            '/usr/local/opt/openjdk',
            '/Library/Java/JavaVirtualMachines/openjdk.jdk/Contents/Home',
            '/usr/lib/jvm/default-java',
            '/usr/lib/jvm/java',
            '/usr/java/latest',
        ];
        for (const p of staticCandidates) {
            if (isValidJavaHome(p)) return p;
        }
        const jvmResult = findHighestJavaInDir('/usr/lib/jvm');
        if (jvmResult) return jvmResult;
    }
    return undefined;
}

// Cache JAVA_HOME discovery
let _cachedJavaHome = null;
let _javaHomeCacheInitialized = false;

function getCachedJavaHome() {
    if (!_javaHomeCacheInitialized) {
        _cachedJavaHome = findJavaHome();
        _javaHomeCacheInitialized = true;
    }
    return _cachedJavaHome;
}

// Cache for expanded paths
let _cachedExpandedPaths = null;
let _cachedPathsInput = null;

function expandPathGlobs(paths) {
    const pathsKey = JSON.stringify(paths);
    if (_cachedExpandedPaths && _cachedPathsInput === pathsKey) {
        return _cachedExpandedPaths;
    }
    const expanded = [];
    for (const p of paths) {
        if (!p) continue;
        if (p.includes('*')) {
            if (!glob) continue;
            try {
                const matches = glob.sync(p, { silent: true });
                expanded.push(...matches);
            } catch (e) {
                console.warn(`[expandPathGlobs] Glob expansion failed for "${p}": ${e.message}`);
            }
        } else if (fs.existsSync(p)) {
            expanded.push(p);
        }
    }
    _cachedExpandedPaths = expanded;
    _cachedPathsInput = pathsKey;
    return expanded;
}

// Colors for console output
const COLORS = {
    reset: "\x1b[0m",
    green: "\x1b[32m",
    red: "\x1b[31m",
    yellow: "\x1b[33m",
    blue: "\x1b[34m",
    cyan: "\x1b[36m",
    magenta: "\x1b[35m"
};

const WORKSPACE_DIR = process.cwd();

function log(message, color = COLORS.reset) {
    console.log(`${color}${message}${COLORS.reset}`);
}

/**
 * Load and parse learnlytica.json configuration file.
 * Returns the config with defaults applied, or null if not found/invalid.
 */
function loadLearnlyticaConfig(workspaceDir) {
    const configPath = path.join(workspaceDir, CONFIG_FILE);

    if (!fs.existsSync(configPath)) {
        return { error: `Configuration file not found: ${CONFIG_FILE}` };
    }

    try {
        const content = fs.readFileSync(configPath, 'utf8');
        const config = JSON.parse(content);

        // Apply defaults
        const result = { ...DEFAULTS, ...config };

        if (!result.components || !Array.isArray(result.components) || result.components.length === 0) {
            return { error: 'Configuration must have at least one component' };
        }

        // Apply component defaults
        result.components = result.components.map(comp => {
            const withDefaults = { ...COMPONENT_DEFAULTS, ...comp };
            withDefaults.scripts = { ...COMPONENT_DEFAULTS.scripts, ...comp.scripts };
            return withDefaults;
        });

        // Validate required fields
        for (const comp of result.components) {
            if (!comp.id) {
                return { error: 'Each component must have an id' };
            }
            if (!comp.baseLanguage) {
                return { error: `Component '${comp.id}' must have a baseLanguage` };
            }
            if (!comp.scripts.test) {
                return { error: `Component '${comp.id}' must have scripts.test defined` };
            }
        }

        return result;
    } catch (e) {
        if (e instanceof SyntaxError) {
            return { error: `Invalid JSON in ${CONFIG_FILE}: ${e.message}` };
        }
        return { error: `Failed to read ${CONFIG_FILE}: ${e.message}` };
    }
}

/**
 * Parse a version string into comparable parts.
 * E.g., "3.8.1" -> [3, 8, 1]
 */
function parseVersion(versionStr) {
    if (!versionStr) return null;
    const parts = versionStr.replace(/^v/, '').split(/[._-]/).map(p => {
        const num = parseInt(p, 10);
        return isNaN(num) ? 0 : num;
    });
    return parts;
}

/**
 * Compare two version arrays.
 * Returns: negative if a < b, positive if a > b, 0 if equal
 */
function compareVersions(a, b) {
    if (!a && !b) return 0;
    if (!a) return -1;
    if (!b) return 1;
    const maxLen = Math.max(a.length, b.length);
    for (let i = 0; i < maxLen; i++) {
        const va = a[i] || 0;
        const vb = b[i] || 0;
        if (va !== vb) return va - vb;
    }
    return 0;
}

/**
 * Check if a version satisfies a constraint.
 * Supports: >=, >, <=, <, =, ^, ~
 */
function satisfiesVersion(actual, constraint) {
    if (!constraint) return true;

    // Parse constraint
    const match = constraint.match(/^(>=|>|<=|<|=|\^|~)?(.+)$/);
    if (!match) return true;

    const [, operator = '>=', requiredStr] = match;
    const actualParts = parseVersion(actual);
    const requiredParts = parseVersion(requiredStr);

    if (!actualParts || !requiredParts) return true;

    const cmp = compareVersions(actualParts, requiredParts);

    switch (operator) {
        case '>=': return cmp >= 0;
        case '>': return cmp > 0;
        case '<=': return cmp <= 0;
        case '<': return cmp < 0;
        case '=': return cmp === 0;
        case '^': // Major version must match, minor/patch can be higher
            return actualParts[0] === requiredParts[0] && cmp >= 0;
        case '~': // Major and minor must match, patch can be higher
            return actualParts[0] === requiredParts[0] &&
                   actualParts[1] === requiredParts[1] &&
                   cmp >= 0;
        default: return cmp >= 0;
    }
}

/**
 * Build a simple enhanced PATH for finding common tools.
 * This is a lightweight version used during prerequisite checking.
 */
function getBasicEnhancedPath() {
    const isWindows = process.platform === 'win32';
    const pathSeparator = isWindows ? ';' : ':';
    const homeDir = process.env.HOME || process.env.USERPROFILE || '';

    const paths = [process.env.PATH || ''];

    if (isWindows) {
        paths.push('C:\\Program Files\\Java\\jdk*\\bin');
        paths.push('C:\\Program Files\\nodejs');
    } else {
        // Common tool locations on macOS/Linux
        paths.push('/usr/local/bin', '/usr/bin', '/bin');
        paths.push('/opt/homebrew/bin', '/opt/homebrew/opt/openjdk/bin');
        paths.push('/usr/local/opt/openjdk/bin');

        // Java locations
        const javaHome = getCachedJavaHome();
        if (javaHome) {
            paths.push(`${javaHome}/bin`);
        }

        // User-specific paths
        if (homeDir) {
            paths.push(`${homeDir}/.local/bin`);
            paths.push(`${homeDir}/.sdkman/candidates/java/current/bin`);
        }
    }

    return paths.filter(Boolean).join(pathSeparator);
}

/**
 * Get the installed version of a tool.
 * Uses spawnSync to capture both stdout and stderr (java outputs to stderr).
 */
function getToolVersion(toolName) {
    const { spawnSync } = require('child_process');

    const commands = {
        'python': ['python3', '--version'],
        'python3': ['python3', '--version'],
        'node': ['node', '--version'],
        'npm': ['npm', '--version'],
        'pip': ['pip3', '--version'],
        'java': ['java', '-version'],
        'javac': ['javac', '-version'],
        'go': ['go', 'version'],
        'rust': ['rustc', '--version'],
        'rustc': ['rustc', '--version'],
        'cargo': ['cargo', '--version'],
        'ruby': ['ruby', '--version'],
        'uv': ['uv', '--version'],
        'gradle': ['gradle', '--version'],
        'maven': ['mvn', '--version'],
        'php': ['php', '--version'],
        'gcc': ['gcc', '--version'],
        'g++': ['g++', '--version'],
        'clang': ['clang', '--version'],
        'swift': ['swift', '--version'],
        'perl': ['perl', '--version'],
        'dotnet': ['dotnet', '--version'],
        'make': ['make', '--version'],
        'cmake': ['cmake', '--version'],
    };

    const cmd = commands[toolName.toLowerCase()];
    if (!cmd) {
        console.log(`[getToolVersion] Unknown tool: ${toolName}`);
        return null;
    }

    try {
        // Build enhanced PATH to find tools in common locations
        const enhancedPath = getBasicEnhancedPath();

        const result = spawnSync(cmd[0], cmd.slice(1), {
            encoding: 'utf8',
            timeout: 5000,
            env: { ...process.env, PATH: enhancedPath },
            shell: false
        });

        // Combine stdout and stderr (java outputs version to stderr)
        const output = (result.stdout || '') + (result.stderr || '');

        if (result.error) {
            console.log(`[getToolVersion] ${toolName} spawn error:`, result.error.message);
            return null;
        }

        // Extract version number from output
        const versionMatch = output.match(/(\d+\.\d+(?:\.\d+)?)/);
        if (versionMatch) {
            console.log(`[getToolVersion] ${toolName} version: ${versionMatch[1]}`);
            return versionMatch[1];
        } else {
            console.log(`[getToolVersion] ${toolName} - no version found in output:`, output.substring(0, 100));
            return null;
        }
    } catch (e) {
        console.log(`[getToolVersion] ${toolName} exception:`, e.message);
        return null;
    }
}

/**
 * Validate prerequisites for a component.
 * Returns { valid: boolean, warnings: string[], errors: string[] }
 */
function validatePrerequisites(prerequisites) {
    const result = { valid: true, warnings: [], errors: [] };

    for (const prereq of prerequisites) {
        const { name, version, optional } = prereq;
        const installedVersion = getToolVersion(name);

        if (!installedVersion) {
            const message = `${name} is not installed`;
            if (optional) {
                result.warnings.push(`⚠️  ${message} (optional)`);
            } else {
                result.errors.push(`❌ ${message} (required)`);
                result.valid = false;
            }
            continue;
        }

        if (version && !satisfiesVersion(installedVersion, version)) {
            const message = `${name} version ${installedVersion} does not satisfy ${version}`;
            if (optional) {
                result.warnings.push(`⚠️  ${message} (optional)`);
            } else {
                result.errors.push(`❌ ${message} (required)`);
                result.valid = false;
            }
            continue;
        }

        log(`  ✓ ${name} ${installedVersion}`, COLORS.green);
    }

    return result;
}

/**
 * Build the enhanced PATH and environment for spawning processes.
 */
function buildEnhancedEnv(componentEnv = {}) {
    const isWindows = process.platform === 'win32';
    const pathSeparator = isWindows ? ';' : ':';
    const homeDir = process.env.HOME || process.env.USERPROFILE || '';

    const globPaths = (!isWindows && homeDir) ? [
        `${homeDir}/.nvm/versions/node/*/bin`,
        `${homeDir}/.fnm/node-versions/*/installation/bin`,
        `${homeDir}/.local/state/fnm_multishells/*/bin`,
        '/usr/local/n/versions/node/*/bin',
        `${homeDir}/.local/share/mise/installs/node/*/bin`
    ] : [];

    const systemPaths = isWindows ? [
        process.env.PATH || '',
        'C:\\Program Files\\nodejs',
        'C:\\Program Files (x86)\\nodejs'
    ] : [
        process.env.PATH || '',
        '/usr/bin', '/usr/local/bin', '/bin',
        '/opt/homebrew/bin', '/opt/homebrew/sbin', '/usr/local/sbin',
        '/opt/homebrew/opt/openjdk/bin', '/usr/local/opt/openjdk/bin',
        '/Library/Java/JavaVirtualMachines/openjdk.jdk/Contents/Home/bin',
        process.env.N_PREFIX ? `${process.env.N_PREFIX}/bin` : null,
        process.env.CONDA_PREFIX ? `${process.env.CONDA_PREFIX}/bin` : null,
        '/opt/anaconda3/bin', '/opt/miniconda3/bin',
        '/opt/homebrew/opt/node/bin', '/usr/local/opt/node/bin',
        '/opt/homebrew/opt/python/bin', '/usr/local/opt/python/bin'
    ];

    const userPaths = homeDir ? (isWindows ? [
        `${homeDir}\\AppData\\Roaming\\npm`,
        `${homeDir}\\.volta\\bin`
    ] : [
        `${homeDir}/.volta/bin`, `${homeDir}/.nodenv/shims`,
        `${homeDir}/.asdf/shims`, `${homeDir}/.local/share/mise/shims`,
        `${homeDir}/.config/mise/shims`, `${homeDir}/n/bin`,
        `${homeDir}/anaconda3/bin`, `${homeDir}/miniconda3/bin`,
        `${homeDir}/miniforge3/bin`, `${homeDir}/mambaforge/bin`,
        `${homeDir}/.pyenv/shims`, `${homeDir}/.local/bin`, `${homeDir}/bin`
    ]) : [];

    const staticPaths = [...systemPaths, ...userPaths].filter(Boolean);
    const expandedGlobPaths = expandPathGlobs(globPaths);
    const allPaths = [...new Set([...staticPaths, ...expandedGlobPaths])];
    const enhancedPath = allPaths.filter(Boolean).join(pathSeparator);

    const detectedJavaHome = getCachedJavaHome();
    const envAdditions = {
        PATH: enhancedPath,
        AUTOMATED_TEST: 'true',
        NO_INTERACTIVE: 'true',
        CI: 'true',
        ...componentEnv
    };

    if (detectedJavaHome) {
        envAdditions.JAVA_HOME = detectedJavaHome;
    }

    return {
        ...process.env,
        ...envAdditions
    };
}

/**
 * Run a shell command and return a promise.
 */
async function runCommand(command, workingDir, env) {
    if (!command || !command.trim()) {
        return { success: true, skipped: true };
    }

    log(`\n🚀 Executing: ${command}`, COLORS.cyan);
    log(`📁 Working directory: ${workingDir}`, COLORS.blue);

    return new Promise((resolve) => {
        const proc = spawn(command, [], {
            cwd: workingDir,
            shell: true,
            stdio: ['ignore', 'inherit', 'inherit'],
            env
        });

        console.log('[UniversalRunner] Process spawned, PID:', proc.pid);

        proc.on('close', (code) => {
            console.log('[UniversalRunner] Process closed with code:', code);
            resolve({ success: code === 0, code });
        });

        proc.on('error', (err) => {
            console.error('[UniversalRunner] Process error:', err);
            log(`❌ Failed to start command: ${err.message}`, COLORS.red);
            resolve({ success: false, error: err.message });
        });

        const timeoutWarning = setTimeout(() => {
            console.warn('[UniversalRunner] ⚠️  Command has been running for 30+ seconds...');
        }, 30000);

        proc.on('close', () => clearTimeout(timeoutWarning));
    });
}

/**
 * Execute a single component's scripts in order.
 */
async function executeComponent(component, workspaceDir, testReportFile) {
    const { id, baseLanguage, scripts, prerequisites, env: componentEnv } = component;

    log(`\n${'═'.repeat(50)}`, COLORS.magenta);
    log(`📦 Component: ${id} (${baseLanguage})`, COLORS.magenta);
    log(`${'═'.repeat(50)}`, COLORS.magenta);

    // Validate prerequisites
    if (prerequisites && prerequisites.length > 0) {
        log('\n🔍 Checking prerequisites...', COLORS.blue);
        const prereqResult = validatePrerequisites(prerequisites);

        for (const warning of prereqResult.warnings) {
            log(warning, COLORS.yellow);
        }
        for (const error of prereqResult.errors) {
            log(error, COLORS.red);
        }

        if (!prereqResult.valid) {
            log(`\n❌ Prerequisites not met for component '${id}'`, COLORS.red);
            return { success: false, component: id, reason: 'Prerequisites not met' };
        }
    }

    const env = buildEnhancedEnv(componentEnv);
    const scriptOrder = ['preInstall', 'install', 'postInstall', 'build', 'test'];

    for (const scriptName of scriptOrder) {
        const script = scripts[scriptName];
        if (!script || !script.trim()) continue;

        log(`\n📜 Running ${scriptName}...`, COLORS.blue);
        const result = await runCommand(script, workspaceDir, env);

        if (!result.success && !result.skipped) {
            // For test script, we continue even on failure (tests may have failed)
            if (scriptName === 'test') {
                log(`⚠️  Tests completed with exit code ${result.code}`, COLORS.yellow);
            } else {
                log(`❌ Script '${scriptName}' failed for component '${id}'`, COLORS.red);
                return { success: false, component: id, reason: `${scriptName} failed` };
            }
        }
    }

    return { success: true, component: id };
}

/**
 * Execute all components according to pipeline configuration.
 */
async function executePipeline(config, workspaceDir) {
    const { components, pipeline, testReportFile } = config;

    if (pipeline === 'parallel') {
        log('\n🔄 Running components in parallel...', COLORS.blue);
        const results = await Promise.all(
            components.map(comp => executeComponent(comp, workspaceDir, testReportFile))
        );
        return results;
    } else {
        // Sequential (default)
        log('\n🔄 Running components sequentially...', COLORS.blue);
        const results = [];
        for (const comp of components) {
            const result = await executeComponent(comp, workspaceDir, testReportFile);
            results.push(result);
            // Stop on first failure in sequential mode
            if (!result.success) {
                log(`\n⏹️  Stopping pipeline due to failure in component '${comp.id}'`, COLORS.red);
                break;
            }
        }
        return results;
    }
}

/**
 * Find a file recursively in a directory
 */
function findFileRecursively(dir, filename) {
    try {
        const files = fs.readdirSync(dir);
        for (const file of files) {
            const fullPath = path.join(dir, file);
            const stat = fs.statSync(fullPath);
            if (stat.isDirectory() && file !== 'node_modules' && file !== '.git' && file !== '__pycache__') {
                const found = findFileRecursively(fullPath, filename);
                if (found) return found;
            } else if (file === filename) {
                return fullPath;
            }
        }
    } catch (e) {
        return null;
    }
    return null;
}

/**
 * Main entry point
 */
async function main() {
    console.log('[UniversalRunner] ========================================');
    console.log('[UniversalRunner] Starting universal test runner');
    console.log('[UniversalRunner] Workspace:', WORKSPACE_DIR);
    console.log('[UniversalRunner] Platform:', process.platform);
    console.log('[UniversalRunner] ========================================');

    log("🔍 Loading learnlytica.json configuration...", COLORS.blue);

    // Load configuration
    const config = loadLearnlyticaConfig(WORKSPACE_DIR);

    if (config.error) {
        log(`\n❌ ${config.error}`, COLORS.red);
        process.exit(1);
    }

    log(`✅ Configuration loaded successfully`, COLORS.green);
    log(`   Version: ${config.version}`, COLORS.cyan);
    log(`   Components: ${config.components.length}`, COLORS.cyan);
    log(`   Pipeline: ${config.pipeline}`, COLORS.cyan);
    log(`   Test Report: ${config.testReportFile}`, COLORS.cyan);

    // Clean up old report
    const reportPath = path.join(WORKSPACE_DIR, config.testReportFile);
    if (fs.existsSync(reportPath)) {
        try {
            fs.unlinkSync(reportPath);
            log(`🧹 Cleaned up old ${config.testReportFile}`, COLORS.yellow);
        } catch (e) {}
    }

    try {
        // Execute pipeline
        const results = await executePipeline(config, WORKSPACE_DIR);

        // Summary
        log(`\n${'═'.repeat(50)}`, COLORS.cyan);
        log('📊 Execution Summary', COLORS.cyan);
        log(`${'═'.repeat(50)}`, COLORS.cyan);

        let allSuccess = true;
        for (const result of results) {
            if (result.success) {
                log(`  ✅ ${result.component}: Success`, COLORS.green);
            } else {
                log(`  ❌ ${result.component}: Failed (${result.reason})`, COLORS.red);
                allSuccess = false;
            }
        }

        // Check for test report
        // First check workspace root
        let finalReportPath = reportPath;
        if (!fs.existsSync(finalReportPath)) {
            // Search recursively
            const found = findFileRecursively(WORKSPACE_DIR, config.testReportFile);
            if (found) {
                finalReportPath = found;
            }
        }

        if (fs.existsSync(finalReportPath)) {
            log(`\n✅ Test report generated: ${finalReportPath}`, COLORS.green);
            const content = fs.readFileSync(finalReportPath, 'utf8');
            // Parse multiple test output formats:
            // - Java/JS custom: [PASS] and [FAIL]
            // - Jest verbose: ✓ and ✕ (or PASS and FAIL)
            // - Generic: passed/failed counts
            let passCount = (content.match(/\[PASS\]/gi) || []).length;
            let failCount = (content.match(/\[FAIL\]/gi) || []).length;

            // If no [PASS]/[FAIL] found, try Jest format (✓ for pass, ✕ or ✗ for fail)
            if (passCount === 0 && failCount === 0) {
                passCount = (content.match(/✓|√/g) || []).length;
                failCount = (content.match(/✕|✗|×/g) || []).length;
            }

            // If still no results, try Jest summary format "X passed" / "X failed"
            if (passCount === 0 && failCount === 0) {
                const passMatch = content.match(/(\d+)\s+passed/i);
                const failMatch = content.match(/(\d+)\s+failed/i);
                if (passMatch) passCount = parseInt(passMatch[1], 10);
                if (failMatch) failCount = parseInt(failMatch[1], 10);
            }

            log(`📊 Test Results: ${passCount} Passed, ${failCount} Failed`, COLORS.cyan);

            // Copy to workspace root if found elsewhere
            if (finalReportPath !== reportPath) {
                fs.copyFileSync(finalReportPath, reportPath);
                log(`📋 Copied report to workspace root`, COLORS.cyan);
            }
        } else {
            log(`\n⚠️  Test report not found: ${config.testReportFile}`, COLORS.yellow);
            log(`   Please ensure your test script generates this file.`, COLORS.yellow);
        }

        if (!allSuccess) {
            process.exit(1);
        }

    } catch (e) {
        log(`\n💥 Critical Error: ${e.message}`, COLORS.red);
        process.exit(1);
    }
}

main();
